"use client"

import { Button } from "@/components/ui/button"
import { X, Plus, Minus, MessageCircle } from "lucide-react"
import { useCart } from "@/components/cart-context"

interface CartDrawerProps {
  isOpen: boolean
  onClose: () => void
}

export function CartDrawer({ isOpen, onClose }: CartDrawerProps) {
  const { items, updateQuantity, removeItem, getTotalPrice, clearCart } = useCart()

  const handleWhatsAppOrder = () => {
    if (items.length === 0) return

    let message = "¡Hola! Me gustaría hacer el siguiente pedido:\n\n"

    items.forEach((item, index) => {
      message += `${index + 1}. ${item.name} (${item.model})\n`
      message += `   Cantidad: ${item.quantity}\n`
      message += `   Precio: S/ ${item.price.toFixed(2)} c/u\n`
      message += `   Subtotal: S/ ${(item.price * item.quantity).toFixed(2)}\n\n`
    })

    message += `TOTAL: S/ ${getTotalPrice().toFixed(2)}\n\n`
    message += "Por favor, confirmen disponibilidad y tiempo de entrega. ¡Gracias!"

    const encodedMessage = encodeURIComponent(message)
    const whatsappUrl = `https://wa.me/51999888777?text=${encodedMessage}`

    window.open(whatsappUrl, "_blank")
    clearCart()
    onClose()
  }

  if (!isOpen) return null

  return (
    <div className="fixed inset-0 z-50 overflow-hidden">
      <div className="absolute inset-0 bg-black bg-opacity-50" onClick={onClose} />

      <div className="absolute right-0 top-0 h-full w-full max-w-md bg-white shadow-xl">
        <div className="flex h-full flex-col">
          {/* Header */}
          <div className="flex items-center justify-between border-b p-4">
            <h2 className="text-lg font-semibold">Carrito de Compras</h2>
            <Button variant="ghost" size="sm" onClick={onClose}>
              <X className="w-5 h-5" />
            </Button>
          </div>

          {/* Cart Items */}
          <div className="flex-1 overflow-y-auto p-4">
            {items.length === 0 ? (
              <div className="text-center text-muted-foreground py-8">
                <p>Tu carrito está vacío</p>
                <p className="text-sm mt-2">Agrega algunos productos para comenzar</p>
              </div>
            ) : (
              <div className="space-y-4">
                {items.map((item) => (
                  <div key={item.id} className="flex gap-3 border-b pb-4">
                    <img
                      src={item.image || "/placeholder.svg"}
                      alt={item.name}
                      className="w-16 h-16 object-cover rounded-lg"
                    />
                    <div className="flex-1">
                      <h3 className="font-medium text-sm">{item.name}</h3>
                      <p className="text-xs text-muted-foreground">{item.model}</p>
                      <p className="font-semibold text-primary">S/ {item.price.toFixed(2)}</p>

                      <div className="flex items-center gap-2 mt-2">
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => updateQuantity(item.id, item.quantity - 1)}
                          className="w-8 h-8 p-0"
                        >
                          <Minus className="w-3 h-3" />
                        </Button>
                        <span className="w-8 text-center text-sm">{item.quantity}</span>
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => updateQuantity(item.id, item.quantity + 1)}
                          className="w-8 h-8 p-0"
                        >
                          <Plus className="w-3 h-3" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => removeItem(item.id)}
                          className="ml-auto text-destructive hover:text-destructive"
                        >
                          <X className="w-4 h-4" />
                        </Button>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>

          {/* Footer */}
          {items.length > 0 && (
            <div className="border-t p-4 space-y-4">
              <div className="flex justify-between items-center">
                <span className="font-semibold">Total:</span>
                <span className="font-bold text-lg text-primary">S/ {getTotalPrice().toFixed(2)}</span>
              </div>

              <Button onClick={handleWhatsAppOrder} className="w-full bg-[#25d366] hover:bg-[#20b858] text-white">
                <MessageCircle className="w-4 h-4 mr-2" />
                Pedir por WhatsApp
              </Button>

              <Button variant="outline" onClick={clearCart} className="w-full bg-transparent">
                Vaciar Carrito
              </Button>
            </div>
          )}
        </div>
      </div>
    </div>
  )
}
